<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Models\User\UserEnum;
use Erlage\Photogram\Data\Tables\Sys\RestTable;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Data\Tables\Admin\AdminTable;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Sys\ResponseTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Models\Setting\SettingEnum;
use Erlage\Photogram\Data\Tables\User\UserBlockTable;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagTable;
use Erlage\Photogram\Data\Tables\Setting\SettingTable;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Models\Setting\SettingFinder;
use Erlage\Photogram\Data\Tables\Post\PostCommentTable;
use Erlage\Photogram\Data\Tables\Post\PostUserTagTable;
use Erlage\Photogram\Data\Tables\User\UserRecoveryTable;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagPostTable;
use Erlage\Photogram\Data\Tables\Post\PostCommentLikeTable;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Tables\Collection\CollectionTable;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagFollowTable;
use Erlage\Photogram\Data\Tables\Notification\NotificationTable;
use Erlage\Photogram\Data\Tables\User\UserEmailVerificationTable;

final class Settings
{
    const DEFAULTS = array(
        SystemConstants::SETTING_NONE                   => 'none',

        /*
        |--------------------------------------------------------------------------
        | user
        |--------------------------------------------------------------------------
        */

        SystemConstants::SETTING_MAX_LEN_USER_USERNAME  => '50',

        SystemConstants::SETTING_MAX_LEN_USER_USERNAME  => '50',
        SystemConstants::SETTING_MIN_LEN_USER_USERNAME  => '6',

        SystemConstants::SETTING_MAX_LEN_USER_EMAIL     => '250',
        SystemConstants::SETTING_MIN_LEN_USER_EMAIL     => '6',

        SystemConstants::SETTING_MAX_LEN_USER_PASSWORD  => '100',
        SystemConstants::SETTING_MIN_LEN_USER_PASSWORD  => '6',

        SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_NAME  => '100',
        SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_NAME  => '0',

        SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_BIO   => '100',
        SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_BIO   => '0',

        SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_WEB   => '100',
        SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_WEB   => '0',

        SystemConstants::SETTING_MAX_USER_DISPLAY_IMAGE_FILE_SIZE => '10',

        SystemConstants::SETTING_SUPPORTED_USER_DISPLAY_IMAGE_FILE_FORMAT => 'png, jpg, jpeg, webp',

        /*
        |--------------------------------------------------------------------------
        | post
        |--------------------------------------------------------------------------
        */

        SystemConstants::SETTING_MAX_LEN_POST_DISPLAY_CAPTION   => '3000',
        SystemConstants::SETTING_MIN_LEN_POST_DISPLAY_CAPTION   => '0',

        SystemConstants::SETTING_MAX_LEN_POST_DISPLAY_LOCATION  => '100',
        SystemConstants::SETTING_MIN_LEN_POST_DISPLAY_LOCATION  => '0',

        SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM  => '12',
        SystemConstants::SETTING_MIN_POST_DISPLAY_CONTENT_ITEM  => '1',

        SystemConstants::SETTING_MAX_POST_HASHTAG               => '8',
        SystemConstants::SETTING_MIN_POST_HASHTAG               => '0',

        SystemConstants::SETTING_MAX_LEN_POST_HASHTAG           => '35',
        SystemConstants::SETTING_MIN_LEN_POST_HASHTAG           => '3',

        SystemConstants::SETTING_MAX_POST_USER_TAG_PER_ITEM     => '8',
        SystemConstants::SETTING_MIN_POST_USER_TAG_PER_ITEM     => '0',

        SystemConstants::SETTING_MAX_POST_USER_TAG_TOTAL        => '40',
        SystemConstants::SETTING_MIN_POST_USER_TAG_TOTAL        => '0',

        SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM_FILE_SIZE => '5',

        SystemConstants::SETTING_SUPPORTED_POST_DISPLAY_CONTENT_ITEM_FILE_FORMAT => 'png, jpg, jpeg, webp',

        /*
        |--------------------------------------------------------------------------
        | post comment
        |--------------------------------------------------------------------------
        */

        SystemConstants::SETTING_MAX_LEN_POST_COMMENT_DISPLAY_TEXT  => '1500',
        SystemConstants::SETTING_MIN_LEN_POST_COMMENT_DISPLAY_TEXT  => '1',

        /*
        |--------------------------------------------------------------------------
        | collection
        |--------------------------------------------------------------------------
        */

        SystemConstants::SETTING_MAX_LEN_COLLECTION_DISPLAY_TITLE   => '50',
        SystemConstants::SETTING_MIN_LEN_COLLECTION_DISPLAY_TITLE   => '1',

        /**
         * [server only settings] clients are not aware of these settings
         * 
         * settings having prefix ss implies server settings and these are not sent
         * to client no matter what 
         */

        /*
        |--------------------------------------------------------------------------
        | identity
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_TEXT_NAME => 'Photogram',

        ServerConstants::SS_TEXT_MINIMUM_API_VERSION => '0.2',

        ServerConstants::SS_TEXT_API_UPDATE_MESSAGE     => 'Whoa! you\'re missing plenty of new features. Update your app.',
        ServerConstants::SS_TEXT_API_IOS_UPDATE_URL     => 'https://address.to.update.for.ios.devices',
        ServerConstants::SS_TEXT_API_ANDROID_UPDATE_URL => 'https://address.to.update.for.android.devices',

        /*
        |--------------------------------------------------------------------------
        | security related
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_INT_OTP_LIFETIME_IN_HOURS => '2',

        ServerConstants::SS_INT_OTP_LIFETIME_IN_TRIES => '5',

        /*
        |--------------------------------------------------------------------------
        | mailer related
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL  => 'contact@yoursite.com',
        ServerConstants::SS_TEXT_MAILER_NO_REPLY_EMAIL => 'noreply@yoursite.com',

        ServerConstants::SS_BOOL_MAILER_SMTP           => '1',

        ServerConstants::SS_TEXT_MAILER_SMTP_HOST      => 'smtp.gmail.com',

        ServerConstants::SS_INT_MAILER_SMTP_PORT       => '465',

        ServerConstants::SS_TEXT_MAILER_SMTP_USERNAME  => 'email@gmail.com',
        ServerConstants::SS_TEXT_MAILER_SMTP_PASSWORD  => 'gmails acc password',

        /*
        |--------------------------------------------------------------------------
        | email template related
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_TEXT_EMAIL_HEADER => '#e1e1e1',

        ServerConstants::SS_TEXT_EMAIL_FOOTER => '',

        /*
        |--------------------------------------------------------------------------
        | uploads
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_TEXT_PATH_UPLOADS                       => DIR_PATH_ROOT . '/uploads/',

        ServerConstants::SS_TEXT_FILE_NAME_PLACEHOLDER_USER         => 'user.png',
        ServerConstants::SS_TEXT_FILE_NAME_PLACEHOLDER_POST         => 'post.png',
        ServerConstants::SS_TEXT_FILE_NAME_PLACEHOLDER_COLLECTION   => 'collection.png',

        /*
        |--------------------------------------------------------------------------
        | user feeds tweaking/targetting
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_LIKE           => '1',

        ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT        => '1',

        ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT_REPLY  => '1',

        /*
        |--------------------------------------------------------------------------
        | content load limits
        |--------------------------------------------------------------------------
        */

        // collections to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_COLLECTION               => '6',

        // user's relationshipts to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_USER_FOLLOW              => '9',

        // user's pending follow requests to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_USER_FOLLOW_PENDING      => '9',

        // users blocked to load per req
        ServerConstants::SS_INT_LIMIT_LOAD_USER_BLOCK               => '9',

        // posts on hashtag page, per request
        ServerConstants::SS_INT_LIMIT_LOAD_HASHTAG_POST             => '9',

        // posts that contains hashtags followed by user
        ServerConstants::SS_INT_LIMIT_LOAD_HASHTAG_POST_EXPLORE     => '9',

        // post comment likes per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_COMMENT_LIKE        => '9',

        // post comments/replies to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_COMMENT             => '3',

        // post likes to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_LIKE                => '9',

        // posts on saved page, per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_SAVE                => '6',

        // posts to load where user is tagged, usually on profile page
        ServerConstants::SS_INT_LIMIT_LOAD_POST_USER_TAG            => '6',

        // posts to load on news feeds, per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_NEWS_FEEDS          => '6',

        // posts to load on profile page, per request
        ServerConstants::SS_INT_LIMIT_LOAD_POST_PROFILE             => '9',

        // users in search results per request
        ServerConstants::SS_INT_LIMIT_LOAD_USER_SEARCH              => '9',

        // notifications to load per request
        ServerConstants::SS_INT_LIMIT_LOAD_NOTIFICATION             => '9',

        // posts to load in infinite mode
        ServerConstants::SS_INT_LIMIT_LOAD_POST_INFINITE            => '12',

        /*
        |--------------------------------------------------------------------------
        | users related
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_BOOL_USER_EMAIL_VERIFICATION => '0',

        ServerConstants::SS_ENUM_USER_DEFAULT_ACCOUNT_PRIVACY => UserEnum::META_IS_PRIVATE_NO,

        /*
        |--------------------------------------------------------------------------
        | default user's notification settings
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_ENUM_NOTIFICATION_PAUSE_ALL                 => UserMetaPushSettingsDTO::VALUE_OFF,

        ServerConstants::SS_ENUM_NOTIFICATION_LIKES                     => UserMetaPushSettingsDTO::VALUE_FROM_EVERYONE,

        ServerConstants::SS_ENUM_NOTIFICATION_COMMENTS                  => UserMetaPushSettingsDTO::VALUE_FROM_EVERYONE,

        ServerConstants::SS_ENUM_NOTIFICATION_PHOTOS_OF_YOU             => UserMetaPushSettingsDTO::VALUE_FROM_EVERYONE,

        ServerConstants::SS_ENUM_NOTIFICATION_LIKES_ON_PHOTOS_OF_YOU    => UserMetaPushSettingsDTO::VALUE_ON,

        ServerConstants::SS_ENUM_NOTIFICATION_COMMENTS_ON_PHOTOS_OF_YOU => UserMetaPushSettingsDTO::VALUE_ON,

        ServerConstants::SS_ENUM_NOTIFICATION_COMMENT_LIKES_AND_PINS    => UserMetaPushSettingsDTO::VALUE_ON,

        ServerConstants::SS_ENUM_NOTIFICATION_ACCEPTED_FOLLOW_REQUEST   => UserMetaPushSettingsDTO::VALUE_ON,

        /*
        |--------------------------------------------------------------------------
        | image compression
        |--------------------------------------------------------------------------
        */

        ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_RES => '150',

        ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_RES => '400',

        ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_RES => '200',

        ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_QUALITY => '90',

        ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_QUALITY => '90',

        ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_QUALITY => '70',

        /*
        |--------------------------------------------------------------------------
        | storage
        |--------------------------------------------------------------------------
        */

        // active storage

        ServerConstants::SS_ENUM_STORAGE_DISK_USER_IMAGES => SettingEnum::STORAGE_DISK_LOCAL,

        ServerConstants::SS_ENUM_STORAGE_DISK_POST_IMAGES => SettingEnum::STORAGE_DISK_LOCAL,

        ServerConstants::SS_ENUM_STORAGE_DISK_COLLECTION_IMAGES => SettingEnum::STORAGE_DISK_LOCAL,

        // aws s3

        ServerConstants::SS_TEXT_AWS_S3_KEY => '',

        ServerConstants::SS_TEXT_AWS_S3_SECRET => '',

        ServerConstants::SS_TEXT_AWS_S3_BUCKET => '',

        ServerConstants::SS_TEXT_AWS_S3_REGION => '',

        // azure blob

        ServerConstants::SS_TEXT_AZURE_BLOB_DSN => '',

        ServerConstants::SS_TEXT_AZURE_BLOB_CONTAINER => '',

        // wasabi s3

        ServerConstants::SS_TEXT_WASABI_S3_KEY => '',

        ServerConstants::SS_TEXT_WASABI_S3_SECRET => '',

        ServerConstants::SS_TEXT_WASABI_S3_BUCKET => '',

        ServerConstants::SS_TEXT_WASABI_S3_REGION => '',
    );

    /**
     * @var array
     */
    private static $settings = array();

    /**
     * @var array
     */
    private static $schema = array();

    public static function init(): void
    {
        self::initSchemaSettings();

        // set defaults

        self::$settings = self::DEFAULTS;

        // override from database

        $finder = (new SettingFinder()) -> find();

        while ($finder -> canPop())
        {
            $settingModel = $finder -> popModelFromResults();

            self::$settings[$settingModel -> getKey()] = $settingModel -> getValue();
        }
    }

    public static function getInt(string $key): int
    {
        if (isset(self::$settings[$key]))
        {
            return (int) (self::$settings[$key]);
        }

        return (int) (self::DEFAULTS[$key]);
    }

    public static function getString(string $key): string
    {
        if (isset(self::$settings[$key]))
        {
            return (string) (self::$settings[$key]);
        }

        return (string) (self::DEFAULTS[$key]);
    }

    public static function getBool(string $key): bool
    {
        if (isset(self::$settings[$key]))
        {
            return (bool) (self::$settings[$key]);
        }

        return (bool) (self::DEFAULTS[$key]);
    }

    public static function getPublicSettings(): array
    {
        $settingsAsString = array();

        foreach (self::$settings as $key => $value)
        {
            if ( ! System::str_starts_with($key, 'ss_'))
            {
                $settingsAsString[(string) $key] = (string) $value;
            }
        }

        return $settingsAsString;
    }

    /*
    |--------------------------------------------------------------------------
    | schema related
    |--------------------------------------------------------------------------
    */

    public static function schema(): array
    {
        return self::$schema;
    }

    private static function initSchemaSettings(): void
    {
        self::$schema = array(
            RestTable::getTableName()                  => RestTable::erlClass(),

            RequestTable::getTableName()               => RequestTable::erlClass(),
            ResponseTable::getTableName()              => ResponseTable::erlClass(),

            UserTable::getTableName()                  => UserTable::erlClass(),
            UserFollowTable::getTableName()            => UserFollowTable::erlClass(),
            UserBlockTable::getTableName()             => UserBlockTable::erlClass(),
            UserRecoveryTable::getTableName()          => UserRecoveryTable::erlClass(),
            UserEmailVerificationTable::getTableName() => UserEmailVerificationTable::erlClass(),

            PostTable::getTableName()                  => PostTable::erlClass(),
            PostLikeTable::getTableName()              => PostLikeTable::erlClass(),
            PostSaveTable::getTableName()              => PostSaveTable::erlClass(),
            PostCommentTable::getTableName()           => PostCommentTable::erlClass(),
            PostCommentLikeTable::getTableName()       => PostCommentLikeTable::erlClass(),
            PostUserTagTable::getTableName()           => PostUserTagTable::erlClass(),

            HashtagTable::getTableName()               => HashtagTable::erlClass(),
            HashtagPostTable::getTableName()           => HashtagPostTable::erlClass(),
            HashtagFollowTable::getTableName()         => HashtagFollowTable::erlClass(),

            CollectionTable::getTableName()            => CollectionTable::erlClass(),

            NotificationTable::getTableName()          => NotificationTable::erlClass(),

            // administration/site related

            SettingTable::getTableName()               => SettingTable::erlClass(),

            AdminTable::getTableName()                 => AdminTable::erlClass(),
        );
    }
}
